% globals.m
% 
% sets a number of options and system demensions
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
% 
% INPUTS:   NONE
% 
% OUTPUTS:  GLOBALS = a structure with many components regarding model
%               specifics, parameters, variable information, and options
%  
% 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
function GLOBALS = globals()



%% -- SYSTEM DIMENSIONS -- %%
GLOBALS.ny      = 16;        % number of nonpredetermined variables
GLOBALS.nx      = 8;         % number of predetermined variables
GLOBALS.ne      = 6;         % number of shocks
GLOBALS.ns      = 2;         % number of regimes
GLOBALS.nY      = 24;        % number of observables


%% -- ESTIMATION DIMENSIONS -- %%
GLOBALS.estimation.NY       = 6; % number of observables in estimation;
GLOBALS.estimation.NX       = 2*GLOBALS.nx + 2*GLOBALS.ny + GLOBALS.ne; % number of states in estimation;


%% -- PARAMETER OPTIONS -- %%
% -- Calibrated Parameters -- %
GLOBALS.parameters.rrho         = 2;            % risk aversion
GLOBALS.parameters.ppsir        = 0;
GLOBALS.parameters.bbar         = 0;
GLOBALS.parameters.varphi0      = 0;
GLOBALS.parameters.varphi1      = 1;
GLOBALS.parameters.nnu0         = 0;
GLOBALS.parameters.nnu1         = 1;

GLOBALS.parameters.bbeta        = 0.99156;      % discount factor
GLOBALS.parameters.oomega       = 1.846;        % labor supply
GLOBALS.parameters.eeta         = 0.30532;      % capital share in production
GLOBALS.parameters.aalpha       = 0.59268;      % labor share in production
GLOBALS.parameters.ddelta       = 0.0227657;    % depreciation rate
GLOBALS.parameters.pbar         = 1.028;        % relative price of imported goods
GLOBALS.parameters.ebar         = 0.11;         % gov purchases 
GLOBALS.parameters.abar         = 6.982/4;      % tfp level
GLOBALS.parameters.zbar         = 1.006;        % tfp growth
GLOBALS.parameters.loggamma01   = -Inf;
GLOBALS.parameters.loggamma11   = -Inf;

% -- Estimated Parameters -- %
% preferences and production
GLOBALS.parm_ind.iiota          = 1;
GLOBALS.parm_ind.pphi           = 2;
GLOBALS.parm_ind.rbar           = 3;
GLOBALS.parm_ind.kkappa         = 4;

% persistences of exogenous processes
GLOBALS.parm_ind.rrhoa          = 5;
GLOBALS.parm_ind.rrhoz          = 6;
GLOBALS.parm_ind.rrhop          = 7;
GLOBALS.parm_ind.rrhor          = 8;
GLOBALS.parm_ind.rrhoe          = 9;
GLOBALS.parm_ind.rrhod          = 10;

% volatilities of exogenous processes
GLOBALS.parm_ind.ssigmaa        = 11;
GLOBALS.parm_ind.ssigmaz        = 12;
GLOBALS.parm_ind.ssigmap        = 13;
GLOBALS.parm_ind.ssigmar        = 14;
GLOBALS.parm_ind.ssigmae        = 15;
GLOBALS.parm_ind.ssigmad        = 16;

% transition probability parameters
GLOBALS.parm_ind.p01            = 17;
GLOBALS.parm_ind.p10            = 18;


%% -- Variable Indices -- %%
% predetermined variables
GLOBALS.var_ind.x.z         = 4;
GLOBALS.var_ind.x.tot       = 5;

% nonpredetermined variables
GLOBALS.var_ind.y.cons      = 1;
GLOBALS.var_ind.y.invest    = 4;
GLOBALS.var_ind.y.intrate   = 6;
GLOBALS.var_ind.y.mmu       = 9;
GLOBALS.var_ind.y.llambda   = 10;
GLOBALS.var_ind.y.bstar     = 11;
GLOBALS.var_ind.y.output    = 12;
GLOBALS.var_ind.y.byratio   = 13;
GLOBALS.var_ind.y.caratio   = 14;
GLOBALS.var_ind.y.tbyratio  = 15;
GLOBALS.var_ind.y.efpd      = 16;

%% -- SOLUTION OPTIONS -- %%
GLOBALS.FRWZprinttime = 0;

%% -- LISTS OF VARIALBES/SHOCKS/PARAMETERS -- %%

GLOBALS.varlist = { '$K$',              'Capital';
                    '$B$ (level)',      'Debt';
                    '$A$',              'Transitory Technology';
                    '$Z$',              'Permanent Technology'
                    '$P$',              'Terms of Trade';
                    '$r^{*} (pp)$',     'Persistant Interest Rate';
                    '$e$',              'Expenditure';
                    '$d$',              'Preference';
                    '$C$',              'Consumption';
                    '$H$',              'Hours';
                    '$V$',              'Intermediate Inputs';
                    '$I$',              'Investment';
                    '$k$',              'Capital';
                    '$r$ (pp)',         'Country Interest Rate';
                    '$Q$',              'Capital Price';
                    '$W$',              'Wages';
                    '$\mu$',            'MU of Consumption';
                    '$\lambda$ (level)','Multiplier';
                    '$B^{*}$ (level)',  'Debt Cushion';
                    '$Y$',              'Output';
                    '$B/Y$ (level)',    'Debt-to-Output Ratio';
                    '$CA/Y$ (level)',   'Current Account-to-Output Ratio';
                    '$TB/Y$ (level)',   'Trade Balance-to-Output Ratio';
                    '$EFPD$ (level)',   'External Fin Prem on Debt'
                    };
GLOBALS.shocklist = {'$\epsilon_a$',        'TFP Shock';
                     '$\epsilon_z$',        'TFP Growth Shock';
                     '$\epsilon_p$',        'Import Price Shock';
                     '$\epsilon_r$'         'Interest Rate Shock'
                     '$\epsilon_e$'         'Expenditure Shock'
                     '$\epsilon_d$'         'Preference Shock'
                    };

GLOBALS.parmlist = {'$\iota$',              'Capital Adj.',             'N',	10,     5;
                    '$\phi$',               'Working Cap.',             'U',     0,     1;
                    '$\bar{r}$',            'Int Rate Mean',            'N',0.0177, 0.005;
                    '$\kappa^{*}$',         'Leverage',                 'U',     0,     1;
                    '$\rho_{a}$',           'Autocor, TFP',             'B',   0.6,   0.2;
                    '$\rho_{z}$',           'Autocor, TFP Growth',      'B',   0.6,   0.2;
                    '$\rho_{p}$',           'Autocor, Imp Price',       'B',   0.6,   0.2;
                    '$\rho_{r}$',           'Autocor, Persist IR',      'B',   0.6,   0.2;
                    '$\rho_{e}$',           'Autocor, Exp',             'B',   0.6,   0.2;
                    '$\rho_{d}$',           'Autocor, Pref',            'B',   0.6,   0.2;
                    '$\sigma_{a,l}$',       'L SD, TFP',               'IG', 0.005, 0.005;
                    '$\sigma_{z,l}$',       'L SD, TFP Growth',        'IG', 0.005, 0.005;
                    '$\sigma_{p,l}$',       'L SD, Imp Price',         'IG',  0.05,  0.05;
                    '$\sigma_{r,l}$',       'L SD, Trans Int Rate',    'IG', 0.005, 0.005;
                    '$\sigma_{e,l}$',       'L SD, Exp',               'IG',   0.1,   0.1;
                    '$\sigma_{d,l}$',       'L SD, Pref',              'IG',  0.05,  0.05;
                    '$P_{0,1}$',            'Prob Enter Binding',       'U',     0,     1;
                    '$P_{1,0}$',            'Prob Exit Binding',        'U',     0,     1;
                    };                    
GLOBALS.hierarchicalprior = 0;



                            
%% -- DATA OPTIONS -- %          
% -- Loading Data from Excel File -- %
load('../Data/bfor_data'); %#ok<LOAD> 

% -- Set Observables in Estimation -- %
GLOBALS.y = [   ygrow...
                cgrow...
                igrow...
                rint...
                cay...
                invtot
                    ]';
                
% observation errors
mscale = 0.05;
GLOBALS.parameters.sigma_obs_1  = sqrt(mscale*var(ygrow));     % output
GLOBALS.parameters.sigma_obs_2  = sqrt(mscale*var(cgrow));     % consumption
GLOBALS.parameters.sigma_obs_3  = sqrt(mscale*var(igrow));     % investment
GLOBALS.parameters.sigma_obs_4  = sqrt(mscale*var(rint));      % interest rate
GLOBALS.parameters.sigma_obs_5  = sqrt(mscale*var(cay));       % ca/y ratio
GLOBALS.parameters.sigma_obs_6  = sqrt(mscale*var(invtot));   % import prices

% -- Dates
GLOBALS.rrdates         = rrdates;
GLOBALS.dates           = dates;    
                                         
% -- Observation Equation Selection Matrix -- %
NY = GLOBALS.estimation.NY;
NX = GLOBALS.estimation.NX;
nx = GLOBALS.nx;
ny = GLOBALS.ny;

D                                       = zeros(NY,NX);
% output growth
D(1,2*nx+GLOBALS.var_ind.y.output)      = 1;    
D(1,2*nx+ny+GLOBALS.var_ind.y.output)   = -1;
D(1,nx+GLOBALS.var_ind.x.z)             = 1;     
% consumption growth
D(2,2*nx+GLOBALS.var_ind.y.cons)        = 1;    
D(2,2*nx+ny+GLOBALS.var_ind.y.cons)     = -1;
D(2,nx+GLOBALS.var_ind.x.z)             = 1;     
% investment growth
D(3,2*nx+GLOBALS.var_ind.y.invest)      = 1;    
D(3,2*nx+ny+GLOBALS.var_ind.y.invest)   = -1;
D(3,nx+GLOBALS.var_ind.x.z)             = 1;     
% country interest rate
D(4,2*nx+GLOBALS.var_ind.y.intrate)     = 1;
D(4,2*nx+GLOBALS.var_ind.y.efpd)        = 0;
% current account to output ratio
D(5,2*nx+GLOBALS.var_ind.y.caratio)     = 1;    
% terms of trade
D(6,GLOBALS.var_ind.x.tot)              = 1;     
D(6,nx+GLOBALS.var_ind.x.tot)           = -1;     

GLOBALS.estimation.D = D;



% -- Observation Error -- %
E  = diag([ GLOBALS.parameters.sigma_obs_1
            GLOBALS.parameters.sigma_obs_2
            GLOBALS.parameters.sigma_obs_3
            GLOBALS.parameters.sigma_obs_4
            GLOBALS.parameters.sigma_obs_5
            GLOBALS.parameters.sigma_obs_6]);
GLOBALS.estimation.E = E;                            


